// $Author: rgerkin $
// $Rev: 1321 $
// $Date: 2010-05-06 09:22:28 -0700 (Thu, 06 May 2010) $

#pragma rtGlobals=1		// Use modern global access method.

//#include "Neuralynx Analysis"

//	NOTE: This requires version 1.25 or later Neuralynx files which start with a 16,384 byte header.
//	Earlier files lack this header.
// 
//	See http://www.neuralynx.com/download/NeuralynxDataFileFormats.pdf for file format specifications.  
//	 
//	This supports .ntt and .ncs files.  A "Neuralynx" menu item is provided in the Analysis menu which provides panel access to the individual file loading and saving functions.  All of the loading functions 
//	are also accessible from the Data/Load Waves menu.  The types variable throughout these function refer to "ntt" or "ncs" (must not contain a leading period) files.  For .ntt files, the data will be loaded
//	as a data wave, a wave with a "_t" suffix (spike times) and a wave with a "_c" suffix (cluster assignments if SpikeSort3D has been used), a wave with a "_h" suffix (the file header), and a wave with a 
//     "_m" suffix (metadata from the file header).  The .ntt data 
//	wave will have 3 dimensions; each spike gives one row.  There are 'NTTSamples' columns because each spike contains 'NTTSamples' samples per channel, and there are 4 layers, one for each channel.  The panel provides
// 	access to a viewer which allows for review of the loaded spike data.  There are also some rudimentary analysis functions such as spike-time-histograms and PSTHs if stimulus times are known.  
//	Primitive clustering is available via PCA but is not recommended since better tools like SpikeSort3D or MClust are available.  Many of the functions provided here do not apply to .ncs (continous) data, 
//	except for the loaders, since there isn't much more to do with these that one can't do on one's own.  

strconstant NlxFolder="root:Packages:Neuralynx:"

// Information that proceeds each chunk of data in .ncs files.   
constant NCSInfoLength=20 // Total bytes per record (8 timestamp, 4 channel number, 4 sampling frequency, 4 num valid samples).  
constant NCSDataSize=2 // Bytes per point (int16).  
constant NCSSamples=512 // Samples per record.  
constant NCSDataLength=1024 //  Bytes per record. (int16 size * NCSSamples).  

Structure NlxNCSInfo
	uint32 TimeStamp_low
	uint32 TimeStamp_high
	uint32 ChannelNumber
	uint32 SampleFreq
	uint32 NumValidSamples
EndStructure

// Information that proceeds each chunk of data in .ntt files.   
constant NTTInfoLength=48 // Total bytes per record (8 timestamp, 4 acquisition entity, 4 for classified cell number, 4x8 for spike feature)	.
constant NTTDataSize=2 // Bytes per point (int16).  
constant NTTSamples=32 // Samples per record per electrode.  (Tetrode = 4 electrodes).  
constant NTTDataLength=256 // Bytes per record. (int16 size x 'NTTSamples' x 4 electrodes/tetrode).  
Structure NlxNTTInfo
	uint32 TimeStamp_low
	uint32 TimeStamp_high
	uint32 SpikeAcqNumber
	uint32 ClassCellNumber
	uint32 FeatureParams[8]
EndStructure

// Information that proceeds each chunk of data in .nev files.   
constant NEVInfoLength=184 // Total bytes per record (8 timestamp, 4 acquisition entity, 4 for classified cell number, 4x8 for spike feature)	.
//constant NTTDataSize=2 // Bytes per point (int16).  
//constant NTTDataLength=256 // Bytes per record. (int16 x 32 x 4).  
Structure NlxNEVInfo
	int16 nstx
	int16 npkt_id
	int16 npkt_data_size
	uint32 TimeStamp_low
	uint32 TimeStamp_high
	int16 nevent_id
	uint16 nttl // The TTL state (16 bits).  
	int16 ncrc
	int16 ndummy1
	int16 ndummy2
	int32 dnExtra[8]
	uchar EventString_low[64]
	uchar EventString_high[64]
EndStructure

Structure NlxFeature
	char name[20]
	uchar num
	char chan
EndStructure

// Things we want to read out of each file's text header.  
static constant NlxHeaderSize = 16384			// Size of header in version 1.25 and later files
Structure NlxMetaData // Metadata in the header.  
	double SampleFreq 
	double bitVolts
	double fileOpenT[2]
	double fileCloseT[2]
	Struct NlxFeature feature[8]
	double reserved[100]
EndStructure

static constant NlxTimestampScale=1000000 // The timestamp units are in microseconds.  

// Add an item to Igor's Load Waves submenu (Data menu)
Menu "Load Waves"
	SubMenu "Load Neuralynx"
		"Load Neuralynx Binary Continuous File...", LoadNlxBinaryFile("ncs","")
		"Load Neuralynx Binary Tetrode File...", LoadNlxBinaryFile("ntt","")
		"Load Neuralynx Event File...", LoadNlxBinaryFile("nev","")
		"Load All Neuralynx Binary Continuous Files In Folder...", LoadAllNlxBinaryFiles("ncs","")
		"Load All Neuralynx Binary Tetrode Files In Folder...", LoadAllNlxBinaryFiles("ntt","")
		//"Edit Neuralynx Loader Settings", EditNlxDefaultSettings()
	End
End

// For editing the default settings of the Neuralynx loader.  Not yet implemented.  To use non-default settings, use optional variables available in the functions.  
//Function EditNlxDefaultSettings()
//End
//
//static Function GetNlxDefaultSettings(type,channelBaseNameDefault, createTableDefault, createGraphDefault)
//	String type
//	String &channelBaseNameDefault				// Output
//	Variable &createTableDefault						// Output
//	Variable &createGraphDefault					// Output
//	
//	type=type[1,strlen(type)-1]
//	strswitch(type)
//		case "ncs":
//			String default_name="Continuous"
//			break
//		case "ntt":
//			default_name="Tetrode"
//			break
//	endswitch
//	
//	String Nlx_folder="root:Packages:Nlx:"
//	channelBaseNameDefault = StrVarOrDefault(Nlx_folder+type+":channelBaseName", default_name)
//	createTableDefault = NumVarOrDefault(Nlx_folder+type+":createTable", 1)
//	createGraphDefault = NumVarOrDefault(Nlx_folder+type+":createGraph", 1)
//End
//
//static Function SetNlxDefaultSettings(type,channelBaseNameDefault, createTableDefault, createGraphDefault)
//	String type, channelBaseNameDefault
//	Variable createTableDefault, createGraphDefault
//	
//	String curr_folder=GetDataFolder(1)
//	NewDataFolder/O/S root:Packages
//	NewDataFolder/O/S Nlx
//	type=type[1,strlen(type)-1]
//	NewDataFolder /O/S $type
//	String/G channelBaseName = channelBaseNameDefault
//	Variable/G createTable = createTableDefault
//	Variable/G createGraph = createGraphDefault
//	SetDataFolder $curr_folder
//End

// Loads the header of a neuralynx file.  Fills the NlxMetaData structure for use by other functions, creates a Name_h wave containing the binary header.  
Function /WAVE LoadNlxHeader(refNum, type, NlxMetaData)
	Variable refNum
	String type
	Struct NlxMetaData &NlxMetaData		// Outputs
	
	String line,key,featureName
	Variable bitVolts,sampleFreq,month,day,year,hours,minutes,secs,featureNum,chan
	FSetPos refNum, 0
	
	FSkipLines(refNum,2)
	FReadLine refNum,line // Currently unused.  
	sscanf line,"## Time Opened (m/d/y): %d/%d/%d  At Time: %d:%d:%f ",month,day,year,hours,minutes,secs
	NlxMetaData.fileOpenT[0]=date2secs(year,month,day)
	NlxMetaData.fileOpenT[1]=3600*hours+60*minutes+secs

	FReadLine refNum,line // Currently unused.  
	sscanf line,"## Time Closed (m/d/y): %d/%d/%d  At Time: %d:%d:%f ",month,day,year,hours,minutes,secs
	NlxMetaData.fileCloseT[0]=date2secs(year,month,day)
	NlxMetaData.fileCloseT[1]=3600*hours+60*minutes+secs
	
	Variable i=0, feature=0
	Do
		FReadLine refNum, line
		sscanf line,"-%s",key
		strswitch(key)
			case "SamplingFrequency":
				sscanf line,"-SamplingFrequency %f",sampleFreq
				NlxMetaData.sampleFreq=sampleFreq
				break
			case "ADBitVolts":
				strswitch(type)
					case "ncs":
						sscanf line,"-ADBitVolts %f",bitVolts
						break
					case "ntt":
						sscanf line,"-ADBitVolts %f",bitVolts // Even though tetrodes could have 4 settings, they are likely to be the same.  
						break
				endswitch
				NlxMetaData.bitVolts=bitVolts
				break
			case "Feature":
				sscanf line,"-Feature %s %d %d",featureName,featureNum,chan
				NlxMetaData.feature[feature].name=featureName
				NlxMetaData.feature[feature].num=featureNum
				NlxMetaData.feature[feature].chan=chan
				feature+=1
				break
		endswitch
		i+=1
	While(i<50)
	
	FSetPos refNum,0
	Make /free/n=(NlxHeaderSize) /B TempHeader
	FBinRead /B=3 refNum, TempHeader
	return TempHeader
End

// Creates the Name_h string containing key-value pairs with all header information for a file.  
static Function /S KeyValueNlxHeader(refNum,type)
	Variable refNum
	String type
	
	FSetPos refNum, 0
	String info="",line,key,value
	Variable i=0
	Do
		FReadLine refNum, line
		if(strsearch(line,"-",0)==0) // Line starts with a hyphen.  
			sscanf line,"-%[A-Za-z:]%s",key,value
			key=ReplaceString(":",key,"") // Remove any colons in the key.  
			info+=key+":"+value+";"
		endif
		FStatus refNum
	While(V_filePos<NlxHeaderSize)
	FSetPos refNum, 0
	return info
End

// Loads the data (everything after the header) from a Neuralynx file.  
static Function /wave LoadNlxData(refNum, type, baseName, NlxMetaData)
	Variable refNum
	String type,baseName
	Struct NlxMetaData &NlxMetaData
		
	FSetPos refNum, NlxHeaderSize													// Go to the end of the header (start of the data).  
	FStatus refNum
	Variable i,j,bytes=V_logEOF-V_filePos
	strswitch(type)
		case "ncs":
			Variable numRecords=bytes/(NCSInfoLength+NCSDataLength) // Number of 'NCSSamples' point frames of continuous data plus associated meta-information.  
			Struct NlxNCSInfo NCSInfo
			FBinRead/B=3 refNum, NCSInfo						// Read the info for the first frame of this continuous event.  
			Variable startT=(NCSInfo.timestamp_low+(2^32)*NCSInfo.timestamp_high)/NlxTimestampScale // Only useful if there are no interruptions in acquisition.  
			Variable chan=NCSInfo.ChannelNumber
			Variable deltaT=1/NCSInfo.SampleFreq
			break
		case "ntt":
			numRecords = bytes/(NTTInfoLength+NTTDataLength) // Number of distinct tetrode records.  
			Struct NlxNTTInfo NTTInfo
			FBinRead/B=3 refNum, NTTInfo						// Read the info for the first frame of this tetrode event.  
			startT=0 										// Each event has its own timestamp.  
			chan=NTTInfo.SpikeAcqNumber
			deltaT=1/NlxMetaData.SampleFreq // Sampling frequency given in the header but not in the frame info for tetrode data.  
			break
		case "nev":
			numRecords = bytes/(NEVInfoLength) // Number of distinct event records.  
			Struct NlxNEVInfo NEVInfo
			FBinRead/B=3 refNum, NEVInfo						// Read the info for the first frame of this tetrode event.  
			startT=0
			break
	endswitch

	if(!strlen(baseName))
		baseName=type
	endif
	String wave_name= CleanupName(baseName,0)
	Make/o/n=(bytes/numRecords,numRecords) /B/U data // uint8 wave (byte wave).  
	FSetPos refNum,NlxHeaderSize 								// Go back to the beginning of the data.  
	FBinRead/B=3 refNum, Data								// Read the rest of the file.  	
	
	// Now sort the wheat from the chaff (clean up the data wave to remove bytes containing metainformation, and resize/redimension it.)  
	NlxIgorizeData(:,type,numRecords)
	
	Note Data, "TYPE:"+type+";"
	strswitch(type)
		case "ncs":
			SetScale/P x, startT, deltaT, "s", Data
			break
		case "ntt":
			break
	endswitch
	
	return Data
End

// Takes Igor waves of data and times, created by LoadNlxData, and saves it back into a Neuralynx file.  Assumes there will be already be an intact header in the file that will be overwritten.  
// Use this if you have done some processing on the data in Igor, like removing bogus spikes or extracting a subset of the data, and now want to re-export it for analysis with Neuralynx tools.  
static Function /S SaveNlxData(refNum, type, df[,Times,Header])
	Variable refNum
	String type
	dfref df
	Wave Times, Header
	
	if(ParamIsDefault(Times))
		Wave Times=df:Times
	endif
	if(ParamIsDefault(Header))
		Wave Header=df:Header
	endif
	
	strswitch(type)
		case "ncs":
			break
		case "ntt":
			Variable numRecords = numpnts(Times) // Number of distinct tetrode records.  
	endswitch
	
	Make/FREE/n=0 /B/U NlxFormatted // uint8 wave (byte wave).  
	NlxDeIgorizeData(df,Times,type,NlxFormatted)
	FSetPos refNum,0
	FBinWrite /B=3 refNum, Header
	FSetPos refNum,NlxHeaderSize 								// Go back to the beginning of the data.  
	FBinWrite/B=3 refNum, NlxFormatted						// Write the new data.  
End

// All the Neuralynx data after the header consists of a series of frames.  Each frame contains data and metadata.  This converts the old metadata+data (Raw) into just the data (Raw) as well 
// as a wave of cluster numbers (_c suffix) and a wave of times (_t suffix).  
Function NlxIgorizeData(df,type,numRecords)
	dfref df
	String type
	Variable numRecords
		
	Variable i
	//Make /o/n=(numRecords)/I NumValidSamples // Always 512 according to Brian at Nlx.  
	strswitch(type)
		case "ncs":
			Make /o/D/U/n=(numRecords) df:times /wave=Times
			break
		case "ntt":
			Make /o/D/U/n=(numRecords) df:times /wave=Times
			Make /o/I/U/n=(numRecords) df:clusters /wave=Clusters
			Make /o/I/U/n=(numRecords,8) df:features /wave=Features 
			break
		case "nev":
			Make /o/T/n=(numRecords) df:desc /wave=Events
			Make /o/U/n=(numRecords) df:TTL /wave=TTL
			Make /o/D/U/n=(numRecords) df:times /wave=Times
			break
	endswitch
	
	wave Raw=df:Data
	for(i=0;i<numRecords;i+=1)
		Variable timestamp=0
		strswitch(type)
			case "ncs":
				Struct NlxNCSInfo NCSInfo
				StructGet /B=3 NCSInfo, Raw[i]
				timestamp=NCSInfo.timestamp_low+(2^32)*NCSInfo.timestamp_high
				Times[i]=timestamp/NlxTimestampScale
				//NumValidSamples[i]=NCSInfo.NumValidSamples
				break
			case "ntt":
				Struct NlxNTTInfo NTTInfo
				StructGet /B=3 NTTInfo, Raw[i]
				timestamp=NTTInfo.timestamp_low+(2^32)*NTTInfo.timestamp_high
				Times[i]=timestamp/NlxTimestampScale
				Clusters[i]=NTTInfo.ClassCellNumber
				Features[i][]=NTTInfo.FeatureParams[q]
				break
			case "nev":
				Struct NlxNEVInfo NEVInfo
				StructGet /B=3 NEVInfo, Raw[i]
				String eventStr=NEVInfo.EventString_low+NEVInfo.EventString_high
				timestamp=1*NEVInfo.timestamp_low+(2^32)*NEVInfo.timestamp_high
				Events[i]=eventStr
				TTL[i]=NEVInfo.nttl
				Times[i]=timestamp/NlxTimestampScale
				break
		endswitch
	endfor
	
	strswitch(type)
			case "ncs":
				DeletePoints /M=0 0,NCSInfoLength,Raw // Delete the metainformation.  
				Redimension /n=(numpnts(Raw)/NCSDataSize)/E=1/W Raw // Convert into one long wave with the appropriate point size.  
				break
			case "ntt":
				DeletePoints /M=0 0,NTTInfoLength,Raw // Delete the metainformation.  
				Redimension /n=(numpnts(Raw)/NTTDataSize)/E=1/W Raw // Convert into one long wave with the appropriate point size.    
				Duplicate /FREE Raw, NTTTemp
				Redimension /n=(NTTSamples,numRecords,4) Raw // Tetrode number (not sample number) is the first index, so redimensioning in one step doesn't work.  
				Raw[][][]=NTTTemp[(p*4)+(128*q)+r] // Maybe use MatrixOp with WaveMap to improve speed?  No, it only supports 2 dimensional waves.  
				break
			case "nev":
				// Nothing left to do since TTL values and times are already extracted.  If you want to do more with the data you should do it in the previous .nev case.  
				break
	endswitch
	// Raw is now clean!  
End

// The opposite of NlxIgorizeData.  This puts the Data and the Times (but not the clusters) back into a Neuralynx frame format for use with re-exporting Neuralynx data with SaveNeuralynxData.  
Function NlxDeIgorizeData(df,Times,type,NlxFormatted)
	dfref df
	Wave Times,NlxFormatted
	String type
	
	Variable i,j,k,numRecords=numpnts(TimeWave)
	Variable numRows=dimsize(NxFormatted,0)
	Struct NlxMetaData NlxMetaData
	Wave /sdfr=df Data,MetaData
	Wave /I/U /sdfr=df Features,Clusters
	StructGet NlxMetaData MetaData
	//Make/O/n=(bytes/numRecords,numRecords) /B/U $(wave_name) /WAVE=NxFormatted // uint8 wave (byte wave).
	//Make /o/n=(numRecords)/I NumValidSamples // Always 512 according to Brian at Nlx.  
	for(i=0;i<numRecords;i+=1)
		Variable timestamp=0
		strswitch(type)
			case "ncs":
				break
			case "ntt":
				Struct NlxNTTInfo NTTInfo
				NTTInfo.timestamp_low=mod(Times[i]*NlxTimestampScale,2^32)
				NTTInfo.timestamp_high=floor(Times[i]*NlxTimestampScale/(2^32))
//				NTTInfo.SpikeAcqNumber=0 // Ignore this since it doesn't matter.  
				NTTInfo.ClassCellNumber=Clusters[i]  
				for(j=0;j<8;j+=1)
					if(WaveExists(Features))
						NTTInfo.FeatureParams[j]=Features[i][j]
					else
						NTTInfo.FeatureParams[j]=MetaData
					endif
				endfor
				StructPut /B=3 NTTInfo, NxFormatted[i]
				break
			case "nev":
				break
		endswitch
	endfor
	strswitch(type)
			case "ncs":
				break
			case "ntt":  
				Duplicate /FREE DataWave,NTTTemp
				Redimension /n=(numpnts(DataWave))/W NTTTemp
				NTTTemp=Data[floor(mod(p,128)/4)][floor(p/128)][mod(p,4)]/NlxMetaData.bitVolts
				Redimension /n=(NTTDataSize*NTTSamples*4,numRecords)/E=1/B/U NTTTemp
				Redimension /n=(NTTInfoLength+NTTSamples*4*NTTDataSize,numRecords) NlxFormatted
				NlxFormatted[NTTInfoLength,][]=NTTTemp[p-NTTInfoLength][q] // Maybe use MatrixOp with WaveMap to improve speed?    
				break
			case "nev":
				break
	endswitch
	// Clean is now raw!  
End

// Load a Neuralynx file.  
Function /s LoadNlxBinaryFile(type,fileName[,pathName,baseName,createTable,createGraph,downSamp])
	String type					// File type: continuous "ncs" or tetrode "ntt".   
	String fileName				// file name, partial path and file name, or full path or "" for dialog.
	String pathName				// Igor symbolic path name or "" for dialog.
	String baseName				// Base name for the wave for this channel .  
	Variable createTable			// 1 = Yes, 2 = No, 0 for dialog.
	Variable createGraph			// 1 = Yes, 2 = No, 0 for dialog.
	Variable downSamp
	
	dfref currFolder=GetDataFolderDFR()
	downSamp = downsamp>1 ? downsamp : 1
	
	strswitch(type)
		case "ncs":
			break
		case "ntt":
			break
		case "nev":
			break
		default:
			DoAlert 0, "Unknown format: "+type
			return "-100" // Exit now so we don't have to deal with the default case again.  
			break
	endswitch
	
	if(ParamIsDefault(pathName))
		PathInfo NlxPath
		if(V_flag)
			pathName="NlxPath"
		else
			pathName="home"
		endif
	endif

	if(ParamIsDefault(baseName))
		baseName=StrVarOrDefault(NlxFolder+"baseNameDefault",fileName)
	endif
	if(ParamIsDefault(createTable))
		 createTable=NumVarOrDefault(NlxFolder+"createTableDefault",0)
	endif
	if(ParamIsDefault(createGraph))
		createGraph=NumVarOrDefault(NlxFolder+"createGraphDefault",0)
	endif
	
	String message = "Select a Nlx binary ."+type+" file"
	Variable refNum
	Open/R/Z=2/P=$pathName/M=message/T=("."+type) refNum as fileName+"."+type
	// Save outputs from Open in a safe place.
	if (V_flag != 0)
		return num2str(V_flag)			// -1 means user canceled.
	endif
	String fullPath = S_fileName
	String path=RemoveListItem(ItemsInList(fullPath,":")-1,fullPath,":")
	fileName=StringFromList(ItemsInList(fullPath,":")-1,fullPath,":")
	fileName=StringFromList(0,fileName,".")
	if(!strlen(baseName))
		baseName=fileName
	endif
	NewDataFolder /O/S $baseName
	dfref df=GetDataFolderDFR()
	NewPath /O/Q/Z NlxPath path
	//printf "Loading Nlx binary "+type+" data from \"%s\"\r", fullPath
	
	Struct NlxMetaData NlxMetaData
	Duplicate /o LoadNlxHeader(refNum,type,NlxMetaData) header
	
	FStatus refNum
	if(V_logEOF<=16384) // Has only the header or even less.  
		print fullPath+" has no data"
		return "-200"
	else
		wave Data=LoadNlxData(refNum, type, baseName, NlxMetaData)	// Load the sample data
	endif
	
	string /g $"type"=type
	string /g headerValues=KeyValueNlxHeader(refNum,type)
	Make /o metadata // Metadata from the header.  
	StructPut NlxMetaData MetaData // Save the header file in case we need to export a modified Nlx file.  
	
	if(StringMatch(type,"ncs"))
		Variable loadedPoints=numpnts(Data)
		Variable delta=deltax(Data)
		if(downSamp>1)
			delta*=downSamp
			if(loadedPoints>100000000) // Do it this way if the wave is large so we don't run out of memory.  
				Variable offset=leftx(Data)
				Data[0,dimsize(Data,0)/downSamp]=mean(Data,offset+p*delta,offset+(p+1)*delta)
				Redimension /n=(loadedPoints/downSamp,-1) Data
				SetScale /P x,leftx(Data),delta,Data
			else // Otherwise, do it the normal way.  
				Resample /DOWN=(downSamp) Data
			endif
		endif
	
		// Fix time wave for continous data so it is a monotonic representation of the time records.  
		if(StringMatch(type,"ncs"))
			Wave Times // Currently just the times for each block of 512 samples.  
			Duplicate /FREE/D Times,temp
			Redimension /D/n=(loadedPoints/downSamp) Times
			Variable factor=NCSSamples/downsamp
			Times=temp[floor(p/factor)]+(temp[floor(p/factor)+1]-temp[floor(p/factor)])*(mod(p,factor)/factor)
		endif
	endif
	
	strswitch(type)
		case "nev":
			variable numSamples = numpnts(Data)
			break
		default: 
			Redimension/S Data												// Change to floating point so we can represent data in volts.
			Data*=NlxMetaData.bitVolts
			SetScale d, 0, 0, "V", Data										// Note that the data units are volts
			strswitch(type)
				case "ncs":
					numSamples = numpnts(Data)
					break
				case "ntt":
					numSamples=dimsize(Data,1)
					break
			endswitch
			break
	endswitch
	printf "Loaded data for %s (%d samples).\r", GetDataFolder(0), numSamples
	
	if (createTable)
		Edit Data.id
		ModifyTable format[1]=3,digits[1]=6										// Display time in suitable format.
	endif
	if (createGraph)
		NlxViewer(df)
	endif

	Close refNum
	SetDataFolder currFolder

	return getdatafolder(1)
End

// Using Igor components, overwrite the data component of a Neuralynx file.  
Function SaveNlxBinaryFile(type,df[,path,fileName,force])
	String type					// File type: continuous "ncs" or tetrode "ntt".   
	dfref df						// The folder which contains the data to be written to a file.  
	String path					// Igor symbolic path name or "" for dialog.
	String fileName				// file name, partial path and file name, or full path or "" for dialog.
	variable force				// Force overwrite without prompting.  
	
	Close /A
	strswitch(type)
		case "ncs":
			break
		case "ntt":
			break
		default:
			DoAlert 0, "Unknown format: "+type
			return -2 // Exit now so we don't have to deal with the default case again.  
			break
	endswitch
	
	Variable err
	
	if(ParamIsDefault(path))
		path="NlxPath"
		pathinfo $path
		if(!strlen(s_path))
			Do
				NewPath /O/M="Choose a default folder for Nlx files."/Q $path
			While(V_flag)
		endif
	endif
	
	// This puts up a dialog if the pathName and fileName do not specify a file.
	String message = "Choose a Nlx binary "+type+" file to overwrite or specify a new name."
	Variable refNum
	if(ParamIsDefault(fileName))
		fileName=nameofwave(datawave)
	endif
	Open /R/Z=1/P=$path/M=message/T=("."+type) refNum as fileName+"."+type
	variable fileExists=!V_flag
	if(fileExists && !force) // File with this name already exists.  
		DoAlert 1,"Overwrite existing file "+fileName+"?"
		if(V_flag>1)
			Close refNum
			return -3
		endif
	endif
	Open /P=$path/M=message/T=("."+type) refNum as fileName+"."+type
	
	// Save outputs from Open in a safe place.
	err = V_Flag
	String fullPath = S_fileName
	
	if (err==-1) // User cancelled.  
		Close refNum
		return -4
	endif
	
	Printf "Writing Nlx binary "+type+" data in \"%s\"\r", fullPath
	
	wave /sdfr=df Data,Times
	SaveNlxData(refNum, type, Data, Times=Times)	// Load the sample data
	
	Close refNum
	return 0			// Zero signifies no error.	
End

// Loads all the neuralynx files of a given type in a directory.  
Function /s LoadAllNlxBinaryFiles(type,baseName[,pathName,createTable,createGraph,downSamp])
	String type				// File type.  
	String pathName			// Name of an Igor symbolic folder created by Misc->NewPath. "" for dialog.
	String baseName	// Base name for the wave for this channel or "" for dialog.
	Variable createTable		// 1 = Yes, 2 = No, 0 for dialog.
	Variable createGraph		// 1 = Yes, 2 = No, 0 for dialog.
	Variable downSamp
	
	if(ParamIsDefault(pathName) || strlen(pathName) == 0)
		String message = "Select a directory containing Nlx "+type+" files"
		NewPath/O/Q/M=message NlxDataPath		// This displays a dialog in which you can select a folder
		if (V_flag != 0)
			return num2str(V_flag)									// -1 means user canceled
		endif
		pathName = "NlxDataPath"
	endif

	if(ParamIsDefault(createTable))
		 createTable=NumVarOrDefault(NlxFolder+"createTableDefault",0)
	endif
	if(ParamIsDefault(createGraph))
		createGraph=NumVarOrDefault(NlxFolder+"createGraphDefault",0)
	endif
	downSamp=ParamIsDefault(downSamp) ? NumVarOrDefault(NlxFolder+"downsampleNCS",0) : downSamp
	
	String filesLoaded = ""
	String fileName
	Variable i = 0
	Do
		fileName = IndexedFile($pathName, i, "."+type)
		fileName=RemoveEnding(fileName,"."+type)
		if (strlen(fileName) == 0)
			break										// No more files
		endif
		
		string dfName=LoadNlxBinaryFile(type,fileName,pathName=pathName,baseName=baseName,createTable=createTable,createGraph=createGraph,downSamp=downSamp)
		variable err=str2num(dfName) // Try converting the return value into a number.  
		if(numtype(err)==0) // If it is a number... 
			return num2str(err) // ... then it is an error code.  
		endif
		if(strlen(dfName))
			filesLoaded+=dfName+";"
		endif
		i += 1
	While(1)
	
	return filesLoaded
End

Function NlxHeaderCheck()
	ControlInfo /W=NeuralynxPanel dirName
	NewPath /O/Q temppath,S_Value
	Variable i=0
		Do
			Variable j=0
			String dirName=IndexedDir(temppath,i,1)
			Do
				NewPath /O/Q temppath2,dirName
				String fileName=IndexedFile(temppath2,j,".ncs")
				if(strlen(fileName))
					String path=dirName+":"+fileName
					Variable refNum
					Open /R refNum as path
					String header=KeyValueNlxHeader(refNum,"ncs")
					// ----- Put code to check headers here.  -----
					FStatus refNum
					if(NumberByKey("AmpLowCut",header)<1 && V_logEOF>16384)
						print path,V_logEOF
					endif
					// -------------------------------------------------------------
					Close refNum
				endif
				j+=1
			While(strlen(fileName))
			i+=1
		While(strlen(dirName))
	//KeyValueNlxHeader(refNum,type)
End

Function NlxViewer(df)
	dfref df
#if exists("CreateNlxViewer")
	CreateNlxViewer(df) // Fancy viewer included in the Neuralynx Analysis procedure file.   
#else
	SimpleNlxViewer(df) // Crappy viewer to use if the Neuralynx Analysis file is not present.  
#endif	
End

Function SimpleNlxViewer(df[,sparse,yAxisRange])
	dfref df
	variable sparse // Plotting all spikes can take a long time, so set this value to "n" to plot only every "nth" spike.  
	variable yAxisRange // A value for the + and - range of the y axis, in volts.  
	
	svar type=df:type
	wave Data=df:Data
	variable i,j,red,green,blue
	Display /K=1 /N=SimpleViewer as CleanupName(GetDataFolder(0,df),0)
	strswitch(type)
		case "ntt":
			wave Clusters=df:Clusters
			variable clusterMax=wavemax(Clusters)
			ColorTab2Wave Rainbow; wave M_Colors
			for(i=0;i<4;i+=1)
				String axis_name="axis_"+num2str(i)
				String axisT_name="axisT_"+num2str(i)
				Variable numSpikes=dimsize(Data,1)
				for(j=0;j<numSpikes;j+=1)  
					Variable cluster=Clusters[j]
					Variable colorIndex=dimsize(M_Colors,0)*(cluster-1)/clusterMax
					red=M_Colors[colorIndex][0]; green=M_Colors[colorIndex][1]; blue=M_Colors[colorIndex][2]
					if(!sparse || mod(j,sparse)==0)				
						AppendToGraph /c=(red,green,blue) /L=$axis_name /B=$axisT_name Data[][j][i]
					endif
				endfor
				ModifyGraph /Z axisEnab($axis_name)={(i<2)*0.52,0.48+(i<2)*0.5}, freePos($axis_name)={0,$axisT_name}
				ModifyGraph /Z axisEnab($axisT_name)={(mod(i,2)==1)*0.52,0.48+(mod(i,2)==1)*0.5}, freePos($axisT_name)={0,$axis_name}
				ModifyGraph /Z tickUnit($axis_name)=1,prescaleExp($axis_name)=6,btlen=1,fsize=9
			endfor
			break
		case "ncs":
			wave Times=df:Times
			AppendToGraph Data vs Times
			break
	endswitch
	
	// Scale axes.  
	string axes=AxisList("")
	for(i=0;i<itemsinlist(axes);i+=1)
		string axis=stringfromlist(i,axes)
		string info=AxisInfo("",axis)
		string axisType=stringbykey("AXTYPE",info)
		strswitch(axisType)
			case "left":
			case "right":
				if(yAxisRange)
					SetAxis /Z $axis -yAxisRange/1000000,yAxisRange/1000000
				else
					SetAxis /A/Z $axis
				endif
				Label /Z $axis " "
				break
		endswitch
	endfor
End

// ---------------- Begin Neuralynx Panel Functions ---------------------

Menu "Analysis"
	"Neuralynx",/Q,NeuralynxPanel()
End

Function NeuralynxPanel() : Panel
	PauseUpdate; Silent 1		// building window...
	DoWindow /K NeuralynxPanel
	NewPanel /K=1 /W=(577,90,1158,199) as "Neuralynx Analysis"
	GetNlxPath("Choose a data directory")
	
	GroupBox SaveLoad,pos={1,0},size={485,47}
	SetVariable dirName,pos={3,2},size={288,16},proc=NeuralynxPanelSetVariables,title="Directory"
	PathInfo NlxPath
	SetVariable dirName,value= _STR:SelectString(V_flag,"",S_path)
	SetVariable fileName,pos={93,25},size={197,16},title="File",value= _STR:""
	PopupMenu Type,pos={3,23},size={83,21},title="Type"
	PopupMenu Type,mode=1,popvalue="ntt",value= #"\"ntt;ncs;nev\"", proc=NeuralynxPanelPopupMenus
	Button ChooseDir,pos={292,1},size={19,20},proc=NeuralynxPanelButtons,title="..."
	SetVariable Downsample,value=_NUM:0,size={100,20},limits={0,Inf,1},title="Downsample"
	Button ChooseFile,pos={292,24},size={19,19},proc=NeuralynxPanelButtons,title="..."
	Button Load,pos={321,24},size={50,20},proc=NeuralynxPanelButtons,title="Load"
	Button Save,pos={376,24},size={50,20},proc=NeuralynxPanelButtons,title="Save"
	
	GroupBox SaveLoad1,pos={1,51},size={574,53}
	PopupMenu Data,pos={3,53},size={83,21},proc=NeuralynxPanelPopupMenus,title="Data"
	PopupMenu Data,mode=1,value= #"NlxRecordings(CtrlStrValue(\"Type\",win=\"NeuralynxPanel\"))"
	Button View,proc=NeuralynxPanelButtons, title="View"
	SetVariable tStart,size={71,16},title="Start",value= _NUM:0
	SetVariable tStop,size={75,16},title="Stop",value= _NUM:inf
	SetVariable binWidth,size={75,16},title="Width",value= _NUM:1
	CheckBox DisplaySTH,size={45,14},title="Show",value= 0
	Button MakeSTH,size={50,20},proc=NeuralynxPanelButtons,title="STH"
	Button Chop,size={50,20},proc=NeuralynxPanelButtons,title="Chop"
	
	PopupMenu STH,pos={3,79},size={97,21},proc=NeuralynxPanelPopupMenus,value=#"\";\"+WaveList(\"*_sth\",\";\",\"\")",title="STH"
	PopupMenu Events,size={87,21},proc=NeuralynxPanelPopupMenus,title="Trig"
	PopupMenu Events,mode=1,value=#"NlxEventList()"
	SetVariable Divisor,size={50,16}, title="mod", value=_NUM:1
	SetVariable Remainder,size={25,16}, value=_NUM:0
	SetVariable tPre,size={65,16},title="Pre",value= _NUM:1
	SetVariable tPost,size={75,16},title="Post",value= _NUM:10
	CheckBox DisplayPSTH,size={45,14},title="Show",value= 0
	Button MakePSTH,size={50,20},proc=NeuralynxPanelButtons,title="PSTH"
End

Function /s NlxEventList()
	string eventWaves=""
	dfref eventsf=root:Events
	if(!datafolderrefstatus(eventsf))
		return wavelist("*",";","")
	endif
	variable numEventWaves=CountObjectsDFR(eventsf,1)
	variable i
	for(i=0;i<numEventWaves;i+=1)
		eventWaves+=getindexedobjnamedfr(eventsf,1,i)+";"
	endfor
	if(strlen(eventWaves))
		eventWaves=";"
	endif
	return eventWaves
End

Function NeuralynxPanelSetVariables(info)
	Struct WMSetVariableAction &info
	if(info.eventCode != 1 && info.eventCode != 2)
		return -1
	endif
	strswitch(info.ctrlName)
		
	endswitch
End

Function NeuralynxPanelButtons(ctrlName)
	String ctrlName
	strswitch(ctrlName)
		case "ChooseDir":
			NewPath /M="Choose a directory of Neuralynx files..."/O/Q/Z NlxPath
			if(!V_flag)
				PathInfo NlxPath
				SetVariable dirName, value=_STR:S_path
			else
				return -2
			endif
			break
		case "ChooseFile":
			Variable refNum
			//PathInfo NeuralynxPath
			ControlInfo Type; String type=S_Value
			Open /D/R/T=("."+type)/Z=2/P=NlxPath/M="Choose a Neuralynx file..." refNum
			if(strlen(S_fileName))
				String fileName=StringFromList(ItemsInList(S_fileName,":")-1,S_fileName,":")
				SetVariable fileName, value=_STR:RemoveEnding(fileName,"."+type)
			endif
			break
		case "Load":
			ControlInfo Type; type=S_Value
			ControlInfo fileName; fileName=S_Value
			ControlInfo DownSample; Variable downSample=V_Value
			ControlInfo dirName; String dirName=S_Value
			NewPath /O/Q/C NeuralynxLoadPath dirName
			string fileNames=PathFiles("NeuralynxLoadPath",extension="."+type)
			fileName=removeending(fileName,"."+type)+"."+type
			string matches=ListMatch(fileNames,fileName)
			variable i
			for(i=0;i<itemsinlist(matches);i+=1)
				fileName=stringfromlist(i,matches)
				fileName=removeending(fileName,"."+type)
				LoadNlxBinaryFile(type,fileName,downSamp=downSample,pathName="NeuralynxLoadPath")
			endfor
			break
		case "Save":
			ControlInfo Type; type=S_Value
			//ControlInfo Data; String dataName=S_Value
			ControlInfo fileName; fileName=S_Value
			ControlInfo dirName; dirName=S_Value
			Wave Data=$fileName
			NewPath /O/Q/C NeuralynxSavePath dirName
			SaveNlxBinaryFile(type,Data,fileName=fileName,path="NeuralynxSavePath")
			break
		case "View":
			ControlInfo Data; dfref df=$S_Value
			NlxViewer(df)
			break
#if exists("NlxSTH")
		case "MakeSTH":
			ControlInfo tStart; Variable tStart=V_Value
			ControlInfo tStop; Variable tStop=V_Value
			ControlInfo Data; Wave Data=$S_Value
			ControlInfo binWidth; Variable binWidth=V_Value
			wave NlxSTH=MakeNlxSTH(Data,binWidth,tStart=tStart,tStop=tStop)
			ControlInfo DisplaySTH
			if(V_Value)
				DisplayNlxSTH(NlxSTH)
			endif
			break
		case "MakePSTH":
			ControlInfo STH; string NlxSTHname=S_Value
			ControlInfo Events; String EventsName=S_Value
			ControlInfo tPre; Variable tPre=V_Value
			ControlInfo tPost; Variable tPost=V_Value
			Wave /Z Events=$EventsName
			if(!WaveExists(Events))
				Wave /Z Events=root:Events:$EventsName
			endif
			if(!WaveExists(Events))
				print "No such wave:"+EventsName
				return -1
			endif
			ControlInfo Divisor; Variable divisor=V_Value
			ControlInfo Remainder; Variable remainder=V_Value
			if(Divisor>1)
				ControlInfo Data; Wave Data=$S_Value
				Wave Events=NlxStimulusTimesByOdor(Data,Divisor)
			endif
			wave NlxPSTH=MakeNlxPSTH(df,Events,tPre,tPost,mode=1)
			ControlInfo DisplayPSTH
			if(V_Value)
				DisplayNlxPSTH(NlxPSTHname,layer=remainder)
			endif
			break
		case "Chop":
			NlxChopWindow()
			break
#endif
	endswitch
End

Function NeuralynxPanelPopupMenus(info)
	Struct WMPopupAction &info
	if(info.eventCode!=2)
		return -1
	endif
	
	strswitch(info.ctrlName)
		case "Data":
			dfref df=$info.popStr
			Wave /sdfr=df data,times
			SetVariable tStart, value=_NUM:wavemin(Times)
			break
		case "Type":
			SetVariable Downsample, disable=!StringMatch(info.popStr,"ncs")
			break
	endswitch
End

Function /S NlxMatchingFolder()
	String fileName=IgorInfo(1)
	GetNlxPath("Choose a data directory")
	Variable i
	Do
		String dirName=IndexedDir(NlxPath,i,0)
		if(!strlen(dirName) || StringMatch(fileName[0,9],ReplaceString("-",dirName[0,9],"_")))
			break
		endif
		i+=1
	While(1)
	return dirName
End

// ------------------------------ Helper functions. -----------------------------

// Skips num_lines lines worth of data from a file.  
static Function FSkipLines(refNum,num_lines)
	Variable refNum,num_lines
	String dummy
	Variable i
	for(i=0;i<num_lines;i+=1)
		FReadLine refNum,dummy
	endfor
	return num_lines
End

static Function /s PathFiles(path[,extension])
	string path,extension
	
	if(paramisdefault(extension))
		extension="????"
	endif
	string files=""
	variable i=0
	do
		string name=IndexedFile($path,i,extension)
		files+=name+";"
		i+=1
	while(strlen(name))
	return removeending(files,";")
End

static Function /S GetNlxPath(message[,path])
	string message
	string path
	
	path=selectstring(paramisdefault(path),path,"NlxPath")
	pathinfo $path
	if(!strlen(S_path))
		newpath /o/q/m=message $path
		pathinfo $path
	endif
	string pathStr=S_path
	return pathstr
End